/*global angular*/
/*jslint node: true */
/*jslint plusplus: true*/

'use strict';

var app = angular.module('readerApp', ['ngCookies']);

app.controller('readerController', function ($scope, $http, $location, $window, $document, $cookies, $timeout) {

    $scope.toggleMenu = function () {
        if (!$scope.displayMenu) {
            $scope.updateMenuFontSelector();
        }
        // don't toggle menu when text is selected
        if ($window.getSelection().toString().length === 0) {
            $scope.displayMenu = !$scope.displayMenu;
        }
        // prevent menu frame from keping focus
        if (!$scope.displayMenu) {
            document.getElementById('menuFrame').blur();
        }
    };

    $scope.closeBook = function () {
        if (sessionStorage.originUrl !== undefined) {
            $window.location.href = sessionStorage.originUrl;
        }
    };

    $scope.goToToc = function () {
        $window.location.href = $scope.tocUrl;
    };

    $scope.handleKey = function (event) {
        var charCode = event.which || event.keyCode,
            navLink;
        if (charCode === 37) {
            // left arrow
            navLink = document.getElementById('prevchapter');
            if (navLink !== null) {
                navLink.click();
            }
        } else if (charCode === 39) {
            // right arrow
            navLink = document.getElementById('nextchapter');
            if (navLink !== null) {
                navLink.click();
            }
        } else if (charCode === 27) {
            // escape
            event.preventDefault(); // http://stackoverflow.com/questions/7218581/esc-key-not-getting-recognized-in-firefox
            $scope.closeBook();
        } else if (charCode === 77) {
            // 'm'
            $scope.toggleMenu();
        } else if (charCode === 32) {
            // space
            if ($window.pageYOffset + $window.innerHeight >= $scope.getDocHeight()) {
                document.getElementById('nextchapter').click();
            }
        }
    };

    $scope.getDocHeight = function () {
        var dcmt = document;
        return Math.max(
            dcmt.body.scrollHeight,
            dcmt.documentElement.scrollHeight,
            dcmt.body.offsetHeight,
            dcmt.documentElement.offsetHeight,
            dcmt.body.clientHeight,
            dcmt.documentElement.clientHeight
        );
    };

    $scope.storeBookmark = function () {
        var bookmark, offset, docHeight, winHeight, percentage, bookmarkString, expireDate = new Date();
        offset = $window.pageYOffset;
        docHeight = $scope.getDocHeight();
        winHeight = $window.innerHeight;

        if (docHeight <= winHeight) {
            percentage = 1;
        } else {
            percentage = offset / (docHeight - winHeight);
        }

        bookmarkString = $scope.spineIndex + '#' + percentage;
        $http.put($scope.buildBookmarkApiUrl() + $scope.bookId, bookmarkString, {
            withCredentials: true
        });       
    };

    $scope.applyBookmark = function (bookmark) {
        var bookmarkArray, percentage;
        bookmarkArray = bookmark.split('#');
        percentage = parseFloat(bookmarkArray[1], 10);
        $scope.gotToPrecentageWhenStyleIsLoaded(percentage, 10); // try 10 times max to load bookmark        
    };

    $scope.getAndApplyBookmarkFromServer = function () {        
        $http.get($scope.buildBookmarkApiUrl() + $scope.bookId, {
            withCredentials: true
        }).then(function (response) {
            if (response.status === 200) {
                $scope.applyBookmark(response.data);
            }
        });
    };

    $scope.buildBookmarkApiUrl = function () {
        var bookmarkUrl = $scope.BOOKMARK_API_URL;
        if ($scope.proxyPrefix && $scope.proxyPrefix !== "") {
            bookmarkUrl = '/' + $scope.proxyPrefix + bookmarkUrl;
        }
        return bookmarkUrl;
    };

    $scope.gotToPrecentageWhenStyleIsLoaded = function (percentage, tries) {
        if (tries > 0) {
            // wait for style to be applid before scrolling to bookmark
            $timeout(function () {
                var marginLeft = document.body.style.marginLeft;
                if (marginLeft) {
                    $scope.goToPercentage(percentage);
                } else {
                    $scope.gotToPrecentageWhenStyleIsLoaded(percentage, tries - 1);
                }
            }, 150);
        }
    };

    $scope.goToPercentage = function (percentage) {
        var targetPos, docHeight, winHeight;
        docHeight = $scope.getDocHeight();
        winHeight = $window.innerHeight;

        if (docHeight > winHeight) {
            targetPos = Math.round(percentage * (docHeight - winHeight));
            $window.scrollTo(0, targetPos);
        }
    };

    $scope.updateFontFace = function (selectedFont, writeCookie) {
        if (selectedFont === 'Default') {
            document.body.style.fontFamily = $scope.originalFontFamily;
        } else {
            document.body.style.setProperty("font-family", selectedFont, "important");
        }
        if (writeCookie) {
            $scope.writeReaderSettings();
        }
    };

    $scope.updateFontSize = function (value, writeCookie) {
        document.body.style.setProperty("font-size", value + '%', "important");
        if (writeCookie) {
            $scope.writeReaderSettings();
        }
    };

    $scope.updateLineSpacing = function (value, writeCookie) {
        document.body.style.setProperty("line-height", value, "important");
        if (writeCookie) {
            $scope.writeReaderSettings();
        }
    };

    $scope.updateMargin = function (value, writeCookie) {
        document.body.style.setProperty("margin-left", value + '%', "important");
        document.body.style.setProperty("margin-right", value + '%', "important");
        if (writeCookie) {
            $scope.writeReaderSettings();
        }
    };

    $scope.setNightMode = function (isNight, writeCookie) {
        if (isNight) {
            document.body.style.setProperty("background-color", "black", "important");
            document.body.style.setProperty("color", "#CCC", "important");
        } else {
            document.body.style.backgroundColor = "";
            document.body.style.color = "";
        }
        if (writeCookie) {
            $scope.writeReaderSettings();
        }
    };

    $scope.resetSettings = function (writeCookie) {
        $scope.fontFace = 'Default';
        $scope.updateFontFace($scope.fontFace, false);
        $scope.fontSize = 100;
        $scope.updateFontSize($scope.fontSize, false);
        $scope.lineSpacing = 1.2;
        $scope.updateLineSpacing("normal", false); // better than 1.2, as default varies from browser to browser
        $scope.margin = 15;
        $scope.updateMargin($scope.margin, false);
        $scope.nightMode = false;
        $scope.setNightMode($scope.nightMode, false);
        if (writeCookie) {
            $scope.writeReaderSettings();
        }
    };

    $scope.writeReaderSettings = function () {
        var data, expireDate = new Date();
        expireDate.setFullYear(expireDate.getFullYear() + 1);
        data = $scope.fontFace + '#' + $scope.fontSize + '#' + $scope.lineSpacing + '#' + $scope.margin + '#' + $scope.nightMode;
        data = btoa(data);
        $cookies.put($scope.READER_SETTINGS, data, {
            'expires': expireDate,
            'samesite': 'lax',
            'path': '/'
        });
    };

    $scope.updateMenuFontSelector = function () {
        var scope = angular.element(document.getElementById('menuFrame')).scope();
        // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
        document.getElementById('menuFrame').contentWindow.updateFontSelector(scope.fontFace);
    }

    ////////////////////
    // INITIALIZATION //
    ////////////////////

    // Constants
    $scope.READER_SETTINGS = 'EpubReaderSettings';
    $scope.BOOKMARK_FINISHED = 'FINISHED';
    $scope.BOOKMARK_API_URL = "/user-api/bookmark?isBook=true&docId=";

    var bookmarkArray, percentage, loadBookmark, readerSettings, readerSettingsArray, computedStyle;

    // keep original font just in case (probably will never happen at body level though)
    computedStyle = $window.getComputedStyle(document.body, null);
    $scope.originalFontFamily = computedStyle.fontFamily;
    $scope.displayMenu = false;
    $scope.spineIndex = parseInt(document.getElementById('spineIndex').innerHTML, 10);
    $scope.bookId = parseInt(document.getElementById('bookId').innerHTML, 10);
    $scope.tocUrl = document.getElementById('tocUrl').innerHTML;
    $scope.proxyPrefix = document.getElementById('proxyPrefix').innerHTML.trim();

    // store current position on close
    $window.onbeforeunload = $scope.storeBookmark;

    $scope.resetSettings(false);
    // Load reader settings
    readerSettings = $cookies.get($scope.READER_SETTINGS);
    if (readerSettings) {
        readerSettingsArray = atob(readerSettings).split('#');
        $scope.fontFace = readerSettingsArray.shift();
        $scope.updateFontFace($scope.fontFace, false);
        $scope.fontSize = parseInt(readerSettingsArray.shift(), 10);
        $scope.updateFontSize($scope.fontSize, false);
        $scope.lineSpacing = parseFloat(readerSettingsArray.shift(), 10);
        $scope.updateLineSpacing($scope.lineSpacing, false);
        $scope.margin = parseInt(readerSettingsArray.shift(), 10);
        $scope.updateMargin($scope.margin, false);
        $scope.nightMode = readerSettingsArray.shift() === "true";
        $scope.setNightMode($scope.nightMode, false);
    }

    // Load bookmark if the book has just been opened (as opposed to just loading a new chapter)
    loadBookmark = sessionStorage.loadBookmark;
    if (loadBookmark) {
        sessionStorage.removeItem('loadBookmark');
        $scope.getAndApplyBookmarkFromServer();
    }

    // END INITIALIZATION
});

app.directive('a', function () {
    return {
        restrict: 'E',
        link: function (scope, elem, attrs) {
            elem.on('click', function (e) {
                e.stopPropagation();
            });
        }
    };
});

/* Utility method */
function clamp(nb, min, max) {
    return Math.min(Math.max(parseFloat(nb), min), max);
}

/* The following functions are proxies to call controller functions from inside the iFrame */

function closeBook() {
    angular.element(document.getElementById('menuFrame')).scope().closeBook();
}

function goToToc() {
    angular.element(document.getElementById('menuFrame')).scope().goToToc();
}

function closeMenu() {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    scope.$apply(function () {
        scope.displayMenu = false;
    });
    document.getElementById('menuFrame').blur();
}

function updateFontFace(selectedFont) {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
    scope.fontFace = selectedFont;
    scope.updateFontFace(selectedFont, true);
}

function modifyFontSize(modValue) {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
    scope.fontSize = clamp(scope.fontSize + modValue, 10, 300);
    scope.updateFontSize(scope.fontSize, true);
}

function modifyLineSpacing(modValue) {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
    scope.lineSpacing = clamp(scope.lineSpacing + modValue, 0.2, 3);
    scope.updateLineSpacing(scope.lineSpacing, true);

}

function modifyMargin(modValue) {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
    scope.margin = clamp(scope.margin + modValue, 0, 40);
    scope.updateMargin(scope.margin, true);
}

function toggleNightMode() {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
    scope.nightMode = !scope.nightMode;
    scope.setNightMode(scope.nightMode, true);
}

function resetSettings() {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    // warning, no apply here, only call function that don"t require it (or enclose in an apply call)
    scope.resetSettings(true);
}

function applyKeyDown(event) {
    var scope = angular.element(document.getElementById('menuFrame')).scope();
    scope.$apply(function () {
        scope.handleKey(event);
    });
}
